# Backend Integration Guide for models.agenticca.ai

## Overview

This document explains how to integrate the UltraWhisper backend API with the models.agenticca.ai proxy infrastructure using NGINX auth_request pattern.

## Architecture

```
Desktop App → models.agenticca.ai/ultrawhisper → NGINX Proxy → AI Providers
                                                        ↓
                                                  Auth Request → Backend API
```

## Integration Points

### 1. Proxy Validation Endpoint

**Backend must implement:** `POST /api/proxy/validate`

This endpoint handles JWT validation and dynamic routing configuration.

#### Request Headers
- `Authorization`: JWT token from desktop application  
- `X-Original-URI`: Original request URI (e.g., `/ultrawhisper/transcribe`)
- `X-Real-IP`: Client IP address

#### Response for Transcription (`/ultrawhisper/transcribe`)

```http
HTTP/200 OK
X-OpenAI-Key: sk-...your-openai-key...

{}
```

#### Response for AI Processing (`/ultrawhisper/process`)

```http  
HTTP/200 OK
X-Target-URL: https://api.openai.com/v1/chat/completions
X-API-Key: sk-...provider-api-key...
X-Current-Model: gpt-4

{}
```

#### Error Response (Invalid/Expired JWT)

```http
HTTP/401 Unauthorized

{"error": "Invalid token"}
```

#### Error Response (Insufficient Credits)

```http
HTTP/403 Forbidden

{"error": "Insufficient credits"}
```

### 2. Dynamic Model Routing

The backend determines which AI provider to route to based on:
- User's selected model in their settings
- Current model availability
- User's subscription/credit status

#### Example Routing Logic

```python
def get_routing_config(user_id: str, current_model: str) -> dict:
    """Determine routing configuration for user's current model"""
    
    # Get user's active model configuration
    user_model = get_user_active_model(user_id)
    
    # Map to provider endpoints
    if user_model.startswith('gpt-'):
        return {
            'target_url': 'https://api.openai.com/v1/chat/completions',
            'api_key': get_openai_key(),
            'current_model': user_model
        }
    elif user_model.startswith('gemini-'):
        return {
            'target_url': f'https://generativelanguage.googleapis.com/v1beta/models/{user_model}:generateContent',
            'api_key': get_google_key(),
            'current_model': user_model
        }
    elif user_model.startswith('claude-'):
        return {
            'target_url': 'https://api.anthropic.com/v1/messages',
            'api_key': get_anthropic_key(),
            'current_model': user_model
        }
    # ... other providers
```

### 3. Request Flow Examples

#### Transcription Request

```bash
# Desktop app makes request
curl -X POST https://models.agenticca.ai/ultrawhisper/transcribe \
  -H "Authorization: Bearer <jwt_token>" \
  -F "file=@audio.wav" \
  -F "model=whisper-1"

# NGINX calls backend for validation
curl -X POST http://backend:8000/api/proxy/validate \
  -H "Authorization: Bearer <jwt_token>" \
  -H "X-Original-URI: /ultrawhisper/transcribe" \
  -H "X-Real-IP: 1.2.3.4"

# Backend responds with OpenAI key
HTTP/200 OK
X-OpenAI-Key: sk-proj-...

# NGINX forwards to OpenAI with injected key
curl -X POST https://api.openai.com/v1/audio/transcriptions \
  -H "Authorization: Bearer sk-proj-..." \
  -F "file=@audio.wav" \
  -F "model=whisper-1"
```

#### AI Processing Request

```bash
# Desktop app makes request
curl -X POST https://models.agenticca.ai/ultrawhisper/process \
  -H "Authorization: Bearer <jwt_token>" \
  -H "Content-Type: application/json" \
  -d '{"messages": [{"role": "user", "content": "Hello"}]}'

# NGINX calls backend for validation + routing
curl -X POST http://backend:8000/api/proxy/validate \
  -H "Authorization: Bearer <jwt_token>" \
  -H "X-Original-URI: /ultrawhisper/process" \
  -H "X-Real-IP: 1.2.3.4"

# Backend responds with routing config
HTTP/200 OK
X-Target-URL: https://api.openai.com/v1/chat/completions
X-API-Key: sk-proj-...
X-Current-Model: gpt-4

# NGINX forwards to configured provider
curl -X POST https://api.openai.com/v1/chat/completions \
  -H "Authorization: Bearer sk-proj-..." \
  -H "Content-Type: application/json" \
  -d '{"model": "gpt-4", "messages": [{"role": "user", "content": "Hello"}]}'
```

## Backend Implementation Requirements

### 1. JWT Token Validation

```python
@app.route('/api/proxy/validate', methods=['POST'])
def validate_proxy_request():
    auth_header = request.headers.get('Authorization')
    original_uri = request.headers.get('X-Original-URI')
    client_ip = request.headers.get('X-Real-IP')
    
    # Validate JWT token
    try:
        token = auth_header.replace('Bearer ', '')
        payload = jwt.decode(token, JWT_SECRET, algorithms=['HS256'])
        user_id = payload['user_id']
        device_id = payload['device_id']
    except:
        return jsonify({'error': 'Invalid token'}), 401
    
    # Check user/device status
    if is_user_blocked(user_id) or is_device_blocked(device_id):
        return jsonify({'error': 'Access denied'}), 403
    
    # Route based on endpoint
    if original_uri == '/ultrawhisper/transcribe':
        return handle_transcribe_validation(user_id)
    elif original_uri == '/ultrawhisper/process':
        return handle_process_validation(user_id)
    else:
        return jsonify({'error': 'Unknown endpoint'}), 404
```

### 2. Credit/Usage Tracking

```python
def handle_transcribe_validation(user_id: str):
    # Check if user has sufficient credits for transcription
    if not has_transcription_credits(user_id):
        return jsonify({'error': 'Insufficient credits'}), 402
    
    # Return OpenAI key for Whisper API
    response = make_response('{}')
    response.headers['X-OpenAI-Key'] = get_openai_api_key()
    return response

def handle_process_validation(user_id: str):
    # Check if user has sufficient credits for AI processing  
    if not has_processing_credits(user_id):
        return jsonify({'error': 'Insufficient credits'}), 402
    
    # Get user's active model and routing config
    routing = get_user_routing_config(user_id)
    
    response = make_response('{}')
    response.headers['X-Target-URL'] = routing['target_url']
    response.headers['X-API-Key'] = routing['api_key']
    response.headers['X-Current-Model'] = routing['model']
    return response
```

### 3. API Key Management

```python
def get_openai_api_key() -> str:
    """Get OpenAI API key - rotate if needed"""
    return os.getenv('OPENAI_API_KEY')

def get_google_api_key() -> str:
    """Get Google AI API key"""
    return os.getenv('GOOGLE_AI_API_KEY')

def get_anthropic_api_key() -> str:
    """Get Anthropic API key"""
    return os.getenv('ANTHROPIC_API_KEY')

def get_user_routing_config(user_id: str) -> dict:
    """Get routing configuration for user's active model"""
    user_settings = get_user_settings(user_id)
    active_model = user_settings.get('ai_model', 'gpt-4')
    
    # Map model to provider configuration
    provider_map = {
        'gpt-4': {
            'target_url': 'https://api.openai.com/v1/chat/completions',
            'api_key': get_openai_api_key(),
            'model': 'gpt-4'
        },
        'gemini-pro': {
            'target_url': 'https://generativelanguage.googleapis.com/v1beta/models/gemini-pro:generateContent',
            'api_key': get_google_api_key(),
            'model': 'gemini-pro'
        },
        # ... other models
    }
    
    return provider_map.get(active_model, provider_map['gpt-4'])  # Default to GPT-4
```

## Error Handling

### 1. NGINX Error Responses

NGINX will return these errors when auth_request fails:

- **502 Bad Gateway**: Backend validation endpoint unreachable
- **401 Unauthorized**: JWT token invalid/expired  
- **403 Forbidden**: User/device blocked or insufficient credits
- **500 Internal Server Error**: Configuration error (missing API keys)

### 2. Backend Error Responses

The backend should return appropriate HTTP status codes:

- **200**: Valid request, headers contain routing info
- **401**: Invalid/expired JWT token
- **402**: Insufficient credits (payment required)
- **403**: User/device access denied
- **404**: Unknown endpoint
- **500**: Internal server error

## Security Considerations

### 1. API Key Protection

- API keys never leave the backend infrastructure
- Keys are injected by NGINX proxy, not visible to client
- Rotate keys regularly without client updates required

### 2. Request Validation

- All requests must include valid JWT tokens
- Device fingerprinting prevents token sharing
- Rate limiting applied at NGINX level (IP + JWT based)

### 3. Credit Tracking

- Credits checked on each request before forwarding
- Usage logged for billing and fraud detection
- Block users/devices with suspicious patterns

## Deployment Configuration

### Environment Variables

Backend must set these environment variables:

```bash
# API Keys
OPENAI_API_KEY=sk-proj-...
GOOGLE_AI_API_KEY=...
ANTHROPIC_API_KEY=...

# JWT Secret (must match desktop app)
JWT_SECRET=your-secret-key

# Database connection
DATABASE_URL=postgresql://...
```

### Backend Health Check

The backend must respond to `GET /health` for NGINX health monitoring:

```python
@app.route('/health')
def health_check():
    return jsonify({
        'status': 'healthy',
        'timestamp': datetime.utcnow().isoformat(),
        'services': {
            'database': check_database_connection(),
            'redis': check_redis_connection(),
            'apis': check_external_apis()
        }
    })
```

## Testing

### 1. Validation Endpoint Testing

```bash
# Test JWT validation
curl -X POST http://localhost:8000/api/proxy/validate \
  -H "Authorization: Bearer <valid_jwt>" \
  -H "X-Original-URI: /ultrawhisper/transcribe" \
  -H "X-Real-IP: 127.0.0.1"

# Should return 200 with X-OpenAI-Key header
```

### 2. End-to-End Testing

```bash  
# Test full proxy flow
curl -X POST https://models.agenticca.ai/ultrawhisper/transcribe \
  -H "Authorization: Bearer <valid_jwt>" \
  -F "file=@test_audio.wav" \
  -F "model=whisper-1"

# Should return transcription from OpenAI
```

## Monitoring

### 1. Backend Metrics

Monitor these metrics in the backend:

- `/api/proxy/validate` response times
- JWT validation success/failure rates  
- API key rotation frequency
- Credit consumption patterns
- User/device blocking events

### 2. NGINX Metrics

Monitor these metrics in NGINX:

- Auth request cache hit rates
- Upstream response times
- Error rates by status code
- Request volume by endpoint

### 3. Usage Logging

Log all proxy requests for analysis:

```python
def log_proxy_request(user_id: str, endpoint: str, model: str, credits_used: int):
    log_entry = {
        'timestamp': datetime.utcnow().isoformat(),
        'user_id': user_id,
        'endpoint': endpoint,
        'model': model,
        'credits_used': credits_used,
        'ip_address': request.remote_addr
    }
    
    # Store in database for billing/analytics
    save_usage_log(log_entry)
```