# UltraWhisper Models Proxy

## Overview

This is a secure reverse proxy infrastructure that sits between the UltraWhisper desktop application and AI providers (OpenAI, Google, Anthropic, etc.). It handles:

- **API Key Injection**: Keys injected by backend, never exposed to client
- **Dynamic Routing**: Backend configures routing per request via auth_request
- **JWT Authentication**: Validated through backend API
- **Usage Logging**: Logs all requests for credit management
- **Health Monitoring**: Continuous backend connectivity monitoring

## Architecture

```
Desktop App → models.agenticca.ai/ultrawhisper → NGINX Proxy → AI Providers
                                                      ↓
                                                 Auth Request → Backend API
                                                      ↓
                                                Health Service (monitoring)
```

## Quick Start

### 1. Prerequisites

- CentOS/RHEL 7+ or Ubuntu 20.04+
- Docker & Docker Compose
- Cloudflare account (for tunnel)

### 2. Configuration

Copy `.env.example` to `.env` and configure:

```bash
cp .env.example .env
nano .env
```

Required variables:
- `BACKEND_URL`: Your backend API URL
- `JWT_SECRET`: Secret for JWT validation
- `SERVICE_TOKEN`: Token for backend communication
- `CLOUDFLARE_TUNNEL_TOKEN`: Cloudflare tunnel token

### 3. Installation

```bash
# Make scripts executable
chmod +x scripts/*.sh

# Run setup
./scripts/setup.sh

# Monitor services
./scripts/monitor.sh
```

### 4. Cloudflare Tunnel Setup

1. Create a tunnel in Cloudflare Zero Trust dashboard
2. Configure route: `models.agenticca.ai` → `http://nginx:80`
3. Copy tunnel token to `.env`

## Services

### NGINX (OpenResty)
- **Port**: 80 (internal), exposed via Cloudflare
- **Config**: `nginx/conf.d/ultrawhisper.conf`
- **Logs**: `logs/ultrawhisper.log`

### Health Service
- **Port**: 8080 (health check)
- **Function**: Monitors backend connectivity and service health
- **Config**: `sync-service/config.yaml`

### Log Processor
- **Function**: Analyzes logs for fraud detection
- **Reports**: Sends suspicious activity to backend

## API Endpoints

### Transcription Endpoint
```
POST https://models.agenticca.ai/ultrawhisper/transcribe
```

Headers:
- `Authorization: Bearer <JWT_TOKEN>`

Body: Standard OpenAI Whisper API format (multipart/form-data)

### AI Processing Endpoint  
```
POST https://models.agenticca.ai/ultrawhisper/process
```

Headers:
- `Authorization: Bearer <JWT_TOKEN>`

Body: Standard OpenAI Chat API format (application/json)

### Health Check
```
GET https://models.agenticca.ai/health
```

## Desktop App Integration

Update your desktop app to use the proxy:

```python
from app.modules.models_client import get_models_provider

# Initialize
models = get_models_provider(auth_manager)

# Transcribe audio
text = models.transcribe_audio("audio.wav", language="en")

# Process with AI
result = models.process_with_ai(text, model="gpt-4")
```

## Monitoring

### View Logs
```bash
# All services
docker-compose logs -f

# Specific service
docker-compose logs -f nginx
docker-compose logs -f sync-service
docker-compose logs -f log-processor
```

### Check Status
```bash
# Service status
docker-compose ps

# Main health check
curl http://localhost:8080/health | jq

# Health service status
curl http://localhost:8081/status | jq
```

### Restart Services
```bash
# All services
docker-compose restart

# Specific service
docker-compose restart nginx
```

## Troubleshooting

### Backend Connectivity Issues
1. Check backend is accessible: `curl http://backend:8000/health`
2. Verify `SERVICE_TOKEN` is correct
3. Check health service logs: `docker-compose logs sync-service`

### JWT Validation Failing
1. Verify `JWT_SECRET` matches backend
2. Check token expiration
3. Review nginx logs: `tail -f logs/ultrawhisper.log`

### High Latency
1. Check backend response times: `curl -w "@curl-format.txt" http://backend:8000/api/proxy/validate`
2. Monitor NGINX performance: `docker stats models-nginx`
3. Check auth cache hit rates in NGINX logs
4. Check network to AI providers

### Fraud Alerts
1. Review log processor output: `docker-compose logs log-processor`
2. Check backend fraud detection logs
3. Verify user/device blocking status via backend API

## Security

### API Keys
- Stored securely in backend only
- Injected by NGINX per request
- Never exposed to client
- Rotated without infrastructure changes

### Authentication
- JWT tokens validated by backend
- NGINX auth_request pattern
- Device fingerprint validation
- User/device blocking via backend API
- Auth responses cached for 30 seconds

### Rate Limiting
- Per-IP: 20 requests/second (burst 10)
- Per-JWT: 100 requests/second (burst 20-50)
- Applied at NGINX level before backend validation

## Maintenance

### Clear Auth Cache
```bash
# Clear NGINX auth cache
curl -X POST http://localhost:8080/clear-cache

# Force backend health check
curl -X POST http://localhost:8081/check
```

### Clean Logs
```bash
# View current log sizes
du -sh logs/

# Clear old logs (keep last 7 days)
find logs/ -name "*.log" -mtime +7 -delete

# Rotate logs manually
docker exec models-nginx nginx -s reopen
```

### Configuration Backup
```bash
# Backup NGINX config
cp nginx/conf.d/ultrawhisper.conf ultrawhisper.conf.backup

# Backup environment
cp .env .env.backup

# Backup docker-compose
cp docker-compose.yml docker-compose.yml.backup
```

## Support

For issues or questions:
1. Check logs first
2. Review this README
3. Contact backend team for API issues
4. Report infrastructure issues to DevOps