-- Usage Logger - Reports usage to backend for credit tracking
local cjson = require "cjson"
local http = require "resty.http"

-- Only log successful requests
if ngx.status ~= 200 then
    return
end

-- Get request metadata
local metadata = ngx.ctx.request_metadata or {}
local jwt_data = ngx.ctx.jwt_data or {}

-- Calculate approximate credits used
local function calculate_credits(action, model, request_time)
    local credits = 0
    
    if action == "transcribe" then
        -- Whisper: ~10 credits per minute of audio
        -- Estimate based on request time (rough approximation)
        credits = math.max(10, math.ceil(request_time * 10))
    elseif action == "ai" then
        -- LLM: varies by model
        if model:match("^gpt%-4") then
            credits = 50
        elseif model:match("^gpt%-3") then
            credits = 20
        elseif model:match("^gemini") then
            credits = 30
        elseif model:match("^claude") then
            credits = 40
        else
            credits = 25  -- Default
        end
    end
    
    return credits
end

-- Prepare usage data
local usage_data = {
    user_id = metadata.user_id or jwt_data.user_id,
    device_id = metadata.device_id or jwt_data.device_id,
    action = metadata.action,
    model = metadata.model,
    provider = metadata.provider,
    language = metadata.language,
    request_id = ngx.var.request_id,
    timestamp = ngx.time(),
    request_time = tonumber(ngx.var.request_time) or 0,
    request_size = tonumber(ngx.var.content_length) or 0,
    response_size = tonumber(ngx.var.bytes_sent) or 0,
    status = ngx.status,
    credits_used = calculate_credits(
        metadata.action,
        metadata.model,
        tonumber(ngx.var.request_time) or 0
    ),
    fingerprint = jwt_data.fingerprint,
    ip_address = ngx.var.remote_addr,
    user_agent = ngx.var.http_user_agent
}

-- Send to backend asynchronously
local function send_usage_to_backend()
    local backend_url = os.getenv("BACKEND_URL") or "http://localhost:8000"
    local httpc = http.new()
    httpc:set_timeout(5000)  -- 5 second timeout
    
    local res, err = httpc:request_uri(
        backend_url .. "/api/usage/register",
        {
            method = "POST",
            body = cjson.encode(usage_data),
            headers = {
                ["Content-Type"] = "application/json",
                ["X-Service-Token"] = os.getenv("SERVICE_TOKEN") or "",
                ["X-Request-ID"] = ngx.var.request_id
            }
        }
    )
    
    if not res then
        ngx.log(ngx.ERR, "Failed to send usage to backend: ", err)
    elseif res.status ~= 200 then
        ngx.log(ngx.WARN, "Backend usage registration failed: ", res.status, " ", res.body)
    else
        ngx.log(ngx.INFO, string.format(
            "Usage logged: user=%s, device=%s, action=%s, credits=%d",
            usage_data.user_id,
            usage_data.device_id,
            usage_data.action,
            usage_data.credits_used
        ))
    end
end

-- Send usage data in a timer (non-blocking)
local ok, err = ngx.timer.at(0, send_usage_to_backend)
if not ok then
    ngx.log(ngx.ERR, "Failed to create timer for usage logging: ", err)
end

-- Log usage data locally for backup (without Redis)
ngx.log(ngx.INFO, "Usage: " .. cjson.encode(usage_data))