#!/bin/bash
# Setup script for UltraWhisper Proxy

set -e

echo "=== UltraWhisper Proxy Setup ==="
echo

# Check requirements
command -v docker >/dev/null 2>&1 || { echo "Docker is required but not installed. Aborting." >&2; exit 1; }
command -v docker-compose >/dev/null 2>&1 || { echo "Docker Compose is required but not installed. Aborting." >&2; exit 1; }

# Create .env file if not exists
if [ ! -f ../.env ]; then
    echo "Creating .env file from template..."
    cp ../.env.example ../.env
    echo "⚠️  Please edit .env file with your configuration"
    echo
fi

# Load environment variables
source ../.env

# Validate required variables
required_vars=("BACKEND_URL" "JWT_SECRET" "SERVICE_TOKEN" "CLOUDFLARE_TUNNEL_TOKEN")
missing_vars=()

for var in "${required_vars[@]}"; do
    if [ -z "${!var}" ]; then
        missing_vars+=($var)
    fi
done

if [ ${#missing_vars[@]} -gt 0 ]; then
    echo "❌ Missing required environment variables:"
    printf ' - %s\n' "${missing_vars[@]}"
    echo
    echo "Please edit .env file and set all required variables"
    exit 1
fi

# Create necessary directories
echo "Creating directories..."
mkdir -p ../logs
chmod 755 ../logs

# Build Docker images
echo "Building Docker images..."
cd ..
docker-compose build

# Start services
echo "Starting services..."
docker-compose up -d

# Wait for services to be healthy
echo "Waiting for services to be healthy..."
sleep 10

# Check health
echo "Checking service health..."
docker-compose ps

# Test NGINX
nginx_health=$(curl -s http://localhost:8080/health | jq -r '.status' 2>/dev/null || echo "error")
if [ "$nginx_health" == "healthy" ] || [ "$nginx_health" == "degraded" ]; then
    echo "✅ NGINX proxy is running"
else
    echo "⚠️  NGINX proxy health check failed"
fi

# Test sync service
sync_health=$(curl -s http://localhost:8080/health | jq -r '.healthy' 2>/dev/null || echo "false")
if [ "$sync_health" == "true" ]; then
    echo "✅ Sync service is running"
else
    echo "⚠️  Sync service health check failed"
fi

echo
echo "=== Setup Complete ==="
echo
echo "Services are running. You can:"
echo "  - View logs: docker-compose logs -f"
echo "  - Check status: docker-compose ps"
echo "  - Stop services: docker-compose down"
echo "  - Restart services: docker-compose restart"
echo
echo "Proxy endpoint: https://models.agenticca.ai/ultrawhisper"
echo "Health check: http://localhost:8080/health"
echo
echo "⚠️  Remember to:"
echo "  1. Configure Cloudflare tunnel to point to this server"
echo "  2. Ensure backend is running at $BACKEND_URL"
echo "  3. Monitor logs for any issues"