#!/bin/bash
# Test script for UltraWhisper Proxy

echo "=== UltraWhisper Proxy Test Suite ==="
echo

# Colors
GREEN='\033[0;32m'
RED='\033[0;31m'
NC='\033[0m'

# Test counter
TESTS_PASSED=0
TESTS_FAILED=0

# Function to run test
run_test() {
    local name="$1"
    local command="$2"
    local expected="$3"
    
    echo -n "Testing $name... "
    
    result=$(eval "$command" 2>/dev/null)
    
    if [[ "$result" == *"$expected"* ]]; then
        echo -e "${GREEN}✓${NC}"
        ((TESTS_PASSED++))
    else
        echo -e "${RED}✗${NC}"
        echo "  Expected: $expected"
        echo "  Got: $result"
        ((TESTS_FAILED++))
    fi
}

# Test 1: NGINX Health
run_test "NGINX health endpoint" \
    "curl -s http://localhost:8080/health | jq -r '.status'" \
    "healthy"

# Test 2: Redis Connection
run_test "Redis connectivity" \
    "docker exec models-redis redis-cli ping" \
    "PONG"

# Test 3: Sync Service Health
run_test "Sync service health" \
    "curl -s http://localhost:8081/health | jq -r '.healthy'" \
    "true"

# Test 4: API Keys in Redis
run_test "API keys loaded in Redis" \
    "docker exec models-redis redis-cli exists api_key:openai" \
    "1"

# Test 5: JWT Validation (negative test)
run_test "JWT validation (no token)" \
    "curl -s -o /dev/null -w '%{http_code}' http://localhost:8080/ultrawhisper" \
    "401"

# Test 6: Rate Limiting Headers
run_test "Rate limiting configured" \
    "curl -s -I http://localhost:8080/health | grep -c 'X-RateLimit'" \
    "0"  # Health endpoint shouldn't have rate limits

# Test 7: Log Files Created
run_test "Log files created" \
    "ls -1 ../logs/*.log 2>/dev/null | wc -l | tr -d ' '" \
    "0"  # Will be >0 after first requests

# Test 8: Docker Services Running
run_test "All Docker services running" \
    "docker-compose ps | grep -c 'Up'" \
    "5"  # nginx, redis, sync-service, cloudflared, log-processor

echo
echo "=== Test Results ==="
echo -e "Passed: ${GREEN}$TESTS_PASSED${NC}"
echo -e "Failed: ${RED}$TESTS_FAILED${NC}"

if [ $TESTS_FAILED -eq 0 ]; then
    echo -e "${GREEN}All tests passed!${NC}"
    exit 0
else
    echo -e "${RED}Some tests failed. Please check the configuration.${NC}"
    exit 1
fi