#!/usr/bin/env python3
"""
Health Check Service for UltraWhisper Models
Provides health status monitoring for the models proxy
"""

import os
import sys
import time
import json
import logging
import threading
from datetime import datetime, timedelta
from typing import Dict, Any, Optional

import requests
from flask import Flask, jsonify

# Configure logging
logging.basicConfig(
    level=os.getenv('LOG_LEVEL', 'INFO'),
    format='%(asctime)s - %(name)s - %(levelname)s - %(message)s'
)
logger = logging.getLogger(__name__)

# Flask app for health checks
app = Flask(__name__)

class HealthCheckService:
    """Health monitoring for UltraWhisper Models proxy"""
    
    def __init__(self):
        self.backend_url = os.getenv('BACKEND_URL', 'http://localhost:8000')
        self.service_token = os.getenv('SERVICE_TOKEN', '')
        
        # State tracking
        self.last_check = None
        self.last_error = None
        self.backend_healthy = False
        
    def check_backend_health(self) -> bool:
        """Check if backend is healthy"""
        try:
            response = requests.get(
                f"{self.backend_url}/health",
                timeout=5
            )
            
            if response.status_code == 200:
                self.backend_healthy = True
                self.last_error = None
                logger.debug("Backend health check successful")
                return True
            else:
                self.backend_healthy = False
                self.last_error = f"Backend returned status {response.status_code}"
                return False
                
        except requests.exceptions.RequestException as e:
            self.backend_healthy = False
            self.last_error = str(e)
            logger.warning(f"Backend health check failed: {e}")
            return False
        except Exception as e:
            self.backend_healthy = False
            self.last_error = str(e)
            logger.error(f"Unexpected error in health check: {e}")
            return False
    
    
    
    
    
    
    def get_status(self) -> Dict[str, Any]:
        """Get service status for health checks"""
        return {
            'healthy': self.backend_healthy and self.last_error is None,
            'last_check': self.last_check.isoformat() if self.last_check else None,
            'last_error': self.last_error,
            'backend_healthy': self.backend_healthy,
            'backend_url': self.backend_url
        }
    
    def run(self):
        """Main health check loop"""
        logger.info("Starting Health Check Service")
        
        # Periodic health checks
        while True:
            try:
                self.check_backend_health()
                self.last_check = datetime.utcnow()
                time.sleep(30)  # Check every 30 seconds
            except KeyboardInterrupt:
                logger.info("Shutting down health service...")
                break
            except Exception as e:
                logger.error(f"Health check error: {e}")
                time.sleep(5)

# Global service instance
service = None

@app.route('/health')
def health():
    """Health check endpoint"""
    if service:
        status = service.get_status()
        return jsonify(status), 200 if status['healthy'] else 503
    return jsonify({'error': 'Service not initialized'}), 503

@app.route('/status')
def status():
    """Detailed status endpoint"""
    if service:
        return jsonify(service.get_status())
    return jsonify({'error': 'Service not initialized'}), 503

@app.route('/check', methods=['POST'])
def trigger_check():
    """Manually trigger health check"""
    if service:
        success = service.check_backend_health()
        service.last_check = datetime.utcnow()
        return jsonify({'success': success}), 200 if success else 500
    return jsonify({'error': 'Service not initialized'}), 503

def run_flask():
    """Run Flask app in separate thread"""
    app.run(host='0.0.0.0', port=8080, debug=False)

if __name__ == "__main__":
    # Initialize service
    service = HealthCheckService()
    
    # Start Flask in background thread
    flask_thread = threading.Thread(target=run_flask, daemon=True)
    flask_thread.start()
    
    # Run health service
    service.run()